from common import *
from pointforces import *

from matplotlib.delaunay.triangulate import delaunay
import numpy.random as rand

def generate_cell(tensL,tensR,setup):
    xMin = setup.xMin
    xMax = setup.xMax
    yMin = setup.yMin
    yMax = setup.yMax
    ds = setup.ds
    
    nX = np.ceil((xMax-xMin)/ds)
    nY = np.ceil((yMax-yMin)/ds)
    
    fL = tensL/nY
    fR = tensR/nY
    gL = 0
    gR = 0
    
    xT, yT = generate_line(xMin,xMax,yMax,yMax,nX)
    xB, yB = generate_line(xMin,xMax,yMin,yMin,nX)
    #xL, yL = generate_line(xMin,xMin,yMin,yMax,nY)
    xR, yR = generate_line(xMax,xMax,yMin,yMax,nY)
    
    pf = PointForces()
    
    pf.add_points(xT, yT, uF = 0, vF = 0)
    pf.add_points(xB, yB, uF = 0, vF = 0)
    #pf.add_points(xL, yL, fF = fL, gF = gL)
    pf.add_points(xR, yR, fF = fR, gF = gR)
    
    xC, yC = generate_circles(setup)
    
    pf.add_points(xC, yC, uF = 0, vF = 0)
    
    return pf

def generate_cell_2(pL,pR,uL,uR,setup):
    xMin = setup.xMin
    xMax = setup.xMax
    yMin = setup.yMin
    yMax = setup.yMax
    ds = setup.ds
    
    nX = np.ceil((xMax-xMin)/ds)
    nY = np.ceil((yMax-yMin)/ds)

    xT, yT = generate_line(xMin,xMax,yMax,yMax,nX)
    xB, yB = generate_line(xMin,xMax,yMin,yMin,nX)
    xL, yL = generate_line(xMin,xMin,yMin,yMax,nY)
    xR, yR = generate_line(xMax,xMax,yMin,yMax,nY)
    
    pf = PointForces()
    
    pf.add_points(xT, yT, uF = 0, vF = 0)
    pf.add_points(xB, yB, uF = 0, vF = 0)
    pf.add_points(xL, yL, pF = pL, uF = 0)
    pf.add_points(xR, yR, pF = pR, uF = 0)
    
    xC, yC = generate_circles(setup)
    
    #pf.add_points(xC, yC, uF = 0, vF = 0)
    
    return pf

def generate_line(xMin,xMax,yMin,yMax,n):
    dx = (np.double(xMax) - np.double(xMin))/n
    dy = (np.double(yMax) - np.double(yMin))/n
    x = np.linspace(np.double(xMin),np.double(xMax),endpoint=False,num=n) + 0.5*dx
    y = np.linspace(np.double(yMin),np.double(yMax),endpoint=False,num=n) + 0.5*dy
    return x, y

def generate_box(xMin,xMax,yMin,yMax,n):
    x_l, y_l = generate_line(xMin,xMin,yMin,yMax,n)
    x_r, y_r = generate_line(xMax,yMax,yMin,yMax,n)
    x_b, y_b = generate_line(xMin,xMax,yMin,yMin,n)
    x_t, y_t = generate_line(xMin,xMax,yMax,yMax,n)
    
    x = np.append(x_l, x_t)
    x = np.append(x, x_r)
    x = np.append(x, x_b)
    y = np.append(y_l, y_t)
    y = np.append(y, y_r)
    y = np.append(y, y_b)
    
    return x, y

# Generate n evenly spaced (in arclength) points around
# the perimeter of a cylinder of radius a center at
# (xC,yC).
def make_cylinder(xC,yC,setup):
    a = setup.a
    n = setup.nFPerCircle
    th = np.linspace(0,2*np.pi,n,endpoint=False)
    x = xC + a*np.cos(th)
    y = yC + a*np.sin(th)
    return x, y

def is_in_box(x,y,box):
    x0, x1, y0, y1 = box
    if x.max() > x1 or x.min() < x0 or y.max() > y1 or y.min() < y0:
        return False
    return True

def is_non_overlapping(x,y,a,tri=None):
    if tri is None:
        tri = delaunay(x,y)
    edges = tri[1]
    dx = x[edges[:,1]] - x[edges[:,0]]
    dy = y[edges[:,1]] - y[edges[:,0]]
    r = np.sqrt(dx**2.0 + dy**2.0)
    rmin = min(r)
    
    if rmin < 2*a:
        return False
    return True

def jiggle(setup,xC,yC,tri=None,box=(0.0,1.0,0.0,1.0)):
    x0, x1, y0, y1 = box
    
    if tri is None:
        tri = delaunay(xC,yC)
    
    a = setup.a
    
    N = len(xC)
    
    circumcenters, edges, tri_points, tri_neighbors = tri
    
    dx = xC[edges[:,1]] - xC[edges[:,0]]
    dy = yC[edges[:,1]] - yC[edges[:,0]]
    r = np.sqrt(dx**2.0 + dy**2.0)

    nx = dx/r
    ny = dy/r
    
    F = 1.0/((r/a)**2.0)

    fX = np.zeros((N))
    fY = np.zeros((N))
    
    for n in range(0,edges.shape[0]):
        e = edges[n]
        fX[e[0]] = fX[e[0]] - F[n]*nx[n]
        fY[e[0]] = fY[e[0]] - F[n]*ny[n]
        
        fX[e[1]] = fX[e[1]] + F[n]*nx[n]
        fY[e[1]] = fY[e[1]] + F[n]*ny[n]
    
    for n in range(0,N):
        if x1 - xC[n] < 2*a:
            fX[n] = fX[n] - 1.0/(((x1-xC[n])/a)**2.0)
        if xC[n] - x0 < 2*a:
            fX[n] = fX[n] + 1.0/(((xC[n]-x0)/a)**2.0)
        if y1 - yC[n] < 2*a:
            fY[n] = fY[n] - 1.0/(((y1-yC[n])/a)**2.0)
        if yC[n] - y0 < 2*a:
            fY[n] = fY[n] + 1.0/(((yC[n]-y0)/a)**2.0)
    
    t = 0.001
    
    xC = xC + t*fX
    yC = yC + t*fY
    
    return xC,yC

def generate_centers_uniformly(setup,num=20,maxIts=400000):
    a = setup.a
    
    LE_dist = 4.0*a
    
    # |-----------------------|-|
    # r                       b l
    
    x_r = 0.0
    x_l = 1.0
    x_b = x_l - LE_dist
    
    def generate():
        if num == 0:
            xC = np.array([])
            yC = np.array([])
        else:
            xC = np.random.rand((num))*(x_b - x_r - 2.0*a) + a
            yC = np.random.rand((num))*(1 - 2.0*a) + a
        return xC, yC

    it = 0    
    xC, yC = generate()
    
    if len(xC) == 0:
        return xC, yC
    
    box = (0.0+a,1.0-LE_dist+a,0.0+a,1.0-a)
    
    while it < maxIts:
        tri = delaunay(xC,yC)
        
        if not is_in_box(xC,yC,box=box):
            xC, yC = generate()
        elif is_non_overlapping(xC,yC,a,tri=tri):
            it = maxIts
        else:
            xC, yC = jiggle(setup,xC,yC,tri=tri,box=box)
        it = it + 1
    
    return xC, yC


# Split the square into two rectangles. One is the front of the cell
# and the other is the remainder of the cell body. Each of these
# should have a uniform distribution of cylinders, but with different
# densities. The density at the front should be much higher than in the
# cell body.
def generate_centers_at_front(setup,body_num=20,front_num=20,maxIts=400000):
    a = setup.a
    
    # Distance at the very leading edge of the cell to where cylinders
    # are allowed.
    LE_dist = 4.0*a
    
    body_fraction = 0.9
    
    # |---------------|-----|---|
    # r               a     b   l
    
    x_r = 0.0
    x_l = 1.0
    x_b = x_l - LE_dist
    x_a = body_fraction*(x_b - x_r) + x_r
    
    def generate_body():
        if body_num == 0:
            xC_body = np.array([])
            yC_body = np.array([])
        else:
            xC_body = np.random.rand((body_num))*(x_a - x_r - 2.0*a) + a
            yC_body = np.random.rand((body_num))*(1 - 2.0*a) + a
        
        #xC_body = xC_body*.6 + .2
        #yC_body = yC_body*.6 + .2
        
        return xC_body, yC_body
    
    def generate_front():
        if front_num == 0:
            xC_front = np.array([])
            yC_front = np.array([])
        else:
            xC_front = np.random.rand((front_num))*(x_b - x_a - 2.0*a) + x_a + a
            yC_front = np.random.rand((front_num))*(1 - 2.0*a) + a
        
        #xC_front = xC_front*.6 + .2
        #yC_front = yC_front*.6 + .2
        
        return xC_front, yC_front
    
    it = 0
    xC_body, yC_body = generate_body()
    xC_front, yC_front = generate_front()
    xC = np.append(xC_body,xC_front)
    yC = np.append(yC_body,yC_front)
    
    if len(xC) == 0:
        return xC, yC
    
    box = (0.0+a,1.0-LE_dist+a,0.0+a,1.0-a)
    
    while it < maxIts:
        tri = delaunay(xC,yC)
        
        
        if not is_in_box(xC,yC,box=box):
            xC_body, yC_body = generate_body()
            xC_front, yC_front = generate_front()
            xC = np.append(xC_body,xC_front)
            yC = np.append(yC_body,yC_front)
        elif is_non_overlapping(xC,yC,a,tri=tri):
            it = maxIts
        else:
            xC, yC = jiggle(setup,xC,yC,tri=tri,box=box)
        it = it + 1
    
    
#    while it < maxIts:
#        tri = delaunay(xC,yC)
#        edges = tri[1]
#        dx = xC[edges[:,0]] - xC[edges[:,1]]
#        dy = yC[edges[:,0]] - yC[edges[:,1]]
#        r = np.sqrt(dx**2.0 + dy**2.0)
#        rmin = min(r)
#        
#        if rmin > 2*a:
#            it = maxIts
#        else:
#            xC, yC = jiggle(setup,xC,yC,tri=tri,box=(x_r,x_a,0.0,1.0))
#        it = it + 1
#        
#    xC_body = xC
#    yC_body = yC
#    
#    it = 0
#    xC, yC = generate_front()
#    
#    while it < maxIts:
#        tri = delaunay(xC,yC)
#        edges = tri[1]
#        dx = xC[edges[:,0]] - xC[edges[:,1]]
#        dy = yC[edges[:,0]] - yC[edges[:,1]]
#        r = np.sqrt(dx**2.0 + dy**2.0)
#        rmin = min(r)
#        
#        if rmin > 2*a:
#            it = maxIts
#        else:
#            xC, yC = jiggle(setup,xC,yC,tri=tri,box=(x_a,x_l,0.0,1.0))
#        it = it + 1
#        
#    xC_front = xC
#    yC_front = yC
    
    
    
#    while it < maxIts:
#        xC, yC = generate_body()
#        
#        circumcenters, edges, tri_points, tri_neighbors = delaunay(xC,yC)
#        dx = xC[edges[:,0]] - xC[edges[:,1]]
#        dy = yC[edges[:,0]] - yC[edges[:,1]]
#        dist = np.sqrt(dx**2.0 + dy**2.0)
#        dmin = min(dist)
#        
#        if dmin > 2*a:
#            it = maxIts
#        
#        it = it + 1
#    
#    xC_body = xC
#    yC_body = yC
#    
#    it = 0
#    
#    while it < maxIts:
#        xC, yC = generate_front()
#        
#        circumcenters, edges, tri_points, tri_neighbors = delaunay(xC,yC)
#        dx = xC[edges[:,0]] - xC[edges[:,1]]
#        dy = yC[edges[:,0]] - yC[edges[:,1]]
#        dist = np.sqrt(dx**2.0 + dy**2.0)
#        dmin = min(dist)
#        
#        if dmin > 2*a:
#            it = maxIts
#        
#        it = it + 1
#        
#    xC_front = xC
#    yC_front = yC
      
#    xC = np.append(xC_body,xC_front)
#    yC = np.append(yC_body,yC_front)
    return xC, yC
    

def generate_centers_pow(setup,maxIts=200000,pow=1.0):
    numCircles = setup.numCircles
    a = setup.a
    # Generate a circle which is entirely within the interior of the
    # unit square, and which is at least two cylinder diameters from
    # the leading edge of the cell.
    
    # Distance from cylinder edge to leading edge
    LE_dist = 4.0*a
    
    xC = ((np.random.rand((numCircles)))**(pow))*(1 - (LE_dist + 2.0*a)) + a
    yC = np.random.rand((numCircles))*(1 - 2.0*a) + a
    if maxIts <= 0 or numCircles <= 1:
        return xC, yC
    dmin = 0
    it = 0
    while it < maxIts:
        
        xC = ((np.random.rand((numCircles)))**(pow))*(1 - (LE_dist + 2.0*a)) + a
        yC = np.random.rand((numCircles))*(1 - 2.0*a) + a
        
        if numCircles == 2:
            dmin = sqrt((xC[0]-xC[1])**2.0 + (yC[0]-yC[1])**2.0)
        else:
            circumcenters, edges, tri_points, tri_neighbors = delaunay(xC,yC)
            dx = xC[edges[:,0]] - xC[edges[:,1]]
            dy = yC[edges[:,0]] - yC[edges[:,1]]
            dist = np.sqrt(dx**2.0 + dy**2.0)
            dmin = min(dist)
        if dmin > 2*a:
            return xC, yC
        it += 1
        del(xC,yC,dx,dy,dist,dmin)
    return xC, yC

def generate_centers(setup,generator=generate_centers_pow):
    return generator(setup)

def generate_gridded_circles(setup):
    numCircles = setup.numCircles
    nFPerCircle = setup.nFPerCircle
    a = setup.a
    xF = np.zeros((0))
    yF = np.zeros((0))
    for k in range(0,numCircles):
        xAdd, yAdd = make_cylinder(xC[k],yC[k],setup)
        xF = np.append(xF,xAdd)
        yF = np.append(yF,yAdd)

def generate_circles(setup,body_num=20,front_num=20,withCenters=False,cents=None):
    numCircles = setup.numCircles
    nFPerCircle = setup.nFPerCircle
    a = setup.a
    if cents is None:
        xC, yC = generate_centers_at_front(setup,body_num,front_num)#generate_centers(setup)
        #xC, yC = generate_centers_uniformly(setup,body_num)
    else:
        xC = cents[0]
        yC = cents[1]
    xF = np.zeros((0))
    yF = np.zeros((0))
    for k in range(0,len(xC)):
        xAdd, yAdd = make_cylinder(xC[k],yC[k],setup)
        xF = np.append(xF,xAdd)
        yF = np.append(yF,yAdd)
    if withCenters:
        return xF, yF, xC, yC
    return xF, yF

def generate_circles_from_centers(setup,xC,yC):
    numCircles = setup.numCircles
    nFPerCircle = setup.nFPerCircle
    a = setup.a
    xF = np.zeros((0))
    yF = np.zeros((0))
    for k in range(0,numCircles):
        xAdd, yAdd = make_cylinder(xC[k],yC[k],setup)
        xF = np.append(xF,xAdd)
        yF = np.append(yF,yAdd)
    return xF, yF

def generate_cell_with_velocity(v_cell,setup,body_num=20,front_num=20,empty=False,flip=False,cents=None):
    xMin = setup.xMin
    xMax = setup.xMax
    yMin = setup.yMin
    yMax = setup.yMax
    ds = setup.ds
    
    nX = np.ceil((xMax-xMin)/ds)
    nY = np.ceil((yMax-yMin)/ds)
    
    xT, yT = generate_line(xMin,xMax,yMax,yMax,nX)
    xB, yB = generate_line(xMin,xMax,yMin,yMin,nX)
    xL, yL = generate_line(xMin,xMin,yMin,yMax,nY)
    xR, yR = generate_line(xMax,xMax,yMin,yMax,nY)
    
    pf = PointForces()
    
    # To make velocity boundary condition continuous,
    # we take the velocity of the leading edge from 0
    # at the corner of the domain, and linearly ramp
    # it up to 1 by the time we are 0.2 into the
    # leading or rear edge
    
    uL = np.ones(xL.shape)
    uL[yL <= 0.2] = yL[yL <= 0.2]/.2
    uL[yL >= 0.8] = (1.0 - yL[yL >= 0.8])/.2

    uR = np.ones(xR.shape)
    uR[yR <= 0.2] = yR[yR <= 0.2]/.2
    uR[yR >= 0.8] = (1.0 - yR[yR >= 0.8])/.2
    
    
    if not flip:
        pf.add_points(xT, yT, uF = 0, vF = 0)
        pf.add_points(xB, yB, uF = 0, vF = 0)
        pf.add_points(xL, yL, uF = v_cell, vF = 0)
        pf.add_points(xR, yR, uF = v_cell, vF = 0)
    else:
        pf.add_points(xT, yT, uF = 0, vF = v_cell)
        pf.add_points(xB, yB, uF = 0, vF = v_cell)
        pf.add_points(xL, yL, uF = 0, vF = 0)
        pf.add_points(xR, yR, uF = 0, vF = 0)
    
    if empty is True:
        return pf
    
    if cents is None:
        xCirc, yCirc, xCent, yCent = generate_circles(setup,body_num,front_num,withCenters=True)
    else:
        xCent = cents[0]
        yCent = cents[1]
        xCirc, yCirc = generate_circles(setup,body_num,front_num,cents=cents)
    
    #xCirc, yCirc = generate_centers_at_front(setup)
    
    pf.xCent = xCent
    pf.yCent = yCent
    pf.add_points(xCirc, yCirc, uF = 0, vF = 0)
    
    return pf
