from common import *
from pointforces import *
from generate import *

import scipy.integrate
import scipy.optimize
import scipy.misc

import numpy.random

from matplotlib.delaunay.triangulate import delaunay

class Curve:
    def __init__(self,X,Y,t0,t1,dXdt=None,dYdt=None,d2X=None,d2Y=None):
        self.X = X
        self.Y = Y
        self.t0 = t0
        self.t1 = t1
        
        if dXdt is None:
            self.dXdt = lambda t: numpy.array(scipy.misc.derivative(X,t,dx=0.0001,n=1,order=7),ndmin=1)
        else:
            self.dXdt = dXdt
        
        if dYdt is None:
            self.dYdt = lambda t: numpy.array(scipy.misc.derivative(Y,t,dx=0.0001,n=1,order=7),ndmin=1)
        else:
            self.dYdt = dYdt
            
        if d2X is None:
            self.d2X = lambda t: numpy.array(scipy.misc.derivative(X,t,dx=0.0001,n=2,order=7),ndmin=1)
        else:
            self.d2X = d2X
        if d2Y is None:
            self.d2Y = lambda t: numpy.array(scipy.misc.derivative(Y,t,dx=0.0001,n=2,order=7),ndmin=1)
        else:
            self.d2Y = d2Y
        
        # Arclength integrand
        self.I = lambda t: np.sqrt(self.dXdt(t)**2.0 + self.dYdt(t)**2.0)
        
    # Arclength starting from an arbitrary t0
    def S(self,t,t0):
        if type(t) is not numpy.ndarray or len(t) == 1:
            return scipy.integrate.quad(self.I,t0,t)[0]
        s = numpy.zeros(t.shape)
        s[0] = scipy.integrate.quad(self.I,t0,t[0])[0]
        for i in xrange(1,len(t)):
            s[i] = s[i-1] + scipy.integrate.quad(self.I,t[i-1],t[i])[0]
            #s[i] = scipy.integrate.quad(self.I,t0,t[i])[0]
        return s



    # Minimum distance from points (X,Y) to the curve and a value of t
    # for which the distance from (X,Y) to (X(t), Y(t)) is minimized.
    def D(self,X,Y):
        if type(X) is np.ndarray:
            T, R = map(np.array,zip(*(map(self._D,X,Y))))
            return T, R
        return self._D(X,Y)
    
    # Minimum distance from point (x,y) to the curve
    def _D(self,x,y):
        t0 = self.t0
        t1 = self.t1      
        
        # Distance from (x, y) to the point (X(t), Y(t))
        r = lambda t: np.sqrt((self.X(t) - x)**2.0 + (self.Y(t) - y)**2.0)
        
        # Derivative of this function
        drdt = lambda t: (1.0/np.sqrt((self.X(t) - x)**2.0 + (self.Y(t) - y)**2.0))*((self.X(t) - x)*self.dXdt(t) + (self.Y(t) - y)*self.dYdt(t))

        # Find a zero of this function
        ##t = scipy.optimize.brentq(drdt,self.t0,self.t1)
        #res = scipy.optimize.minimize(r,(0.5*(self.t0+self.t1)),method='L-BFGS-B',jac=drdt,bounds=((self.t0,self.t1)))
        
        bnds = [(t0,t1)]
        t0 = np.array(0.5*(t0+t1),ndmin=1)
        
        res = scipy.optimize.minimize(r,t0,method='L-BFGS-B',bounds=bnds)
        
        
        t = res.x[0]
        
        return t, r(t)
    
    # Closest point (X(t), Y(t)) on the curve to the point (x,y)
    def min_dist(self,x,y):
        t, r = self.D(x,y)
        return self(t)
        

    # Unit tangent vector to curve
    def T(self, t):
        Tx = self.dXdt(t)
        Ty = self.dYdt(t)
        
        mag_T = self.I(t)
        
        Tx_hat = Tx/mag_T
        Ty_hat = Ty/mag_T
        
        return (Tx_hat, Ty_hat)

    # Unit normal vector to curve
    def N(self, t):
        Nx = -self.dYdt(t)
        Ny = self.dXdt(t)
        
        mag_N = self.I(t)
        
        Nx_hat = Nx/mag_N
        Ny_hat = Ny/mag_N
        
        return (Nx_hat, Ny_hat)
    
    def __call__(self,t):
        return (self.X(t),self.Y(t))
    
    def parameterize(self,N,centered=True):
        X = self.X
        Y = self.Y
        dXdt = self.dXdt
        dYdt = self.dYdt
        t0 = self.t0
        t1 = self.t1
        S = lambda t: self.S(t,t0)
        
        # Integrand for the arclength computation
        #I = lambda t: np.sqrt(dXdt(t)**2.0 + dYdt(t)**2.0)
        
        # Arclength S(t) of the parametric curve
        #S = lambda t: (scipy.integrate.quad(I,t0,t))[0]
        
        # Inverse T(s) of the arclength
        def T(s):
            # Function whose root we wish to find
            F = lambda t: S(t) - s
            
            #t = scipy.optimize.newton(F, s, self.I)
            t = scipy.optimize.brentq(F,t0,t1)
            
            return t
        
        # Arclength at endpoints of curve. s0 should be 0
        s0 = S(t0)
        s1 = S(t1)
        
        # Arclength values for the points we will return
        if centered:
            ds = (s1 - s0)/N
            s = np.linspace(s0+0.5*ds,s1-0.5*ds,N,True)
        else:
            s = np.linspace(s0,s1,N)
    
        t = []
        
        for ss in s:
            tt = T(ss)
            t.append(tt)
            
        t = np.array(t)
        
        return (X(t),Y(t),t)
    
    def average_pressure(self,setup,pf,t0=None,t1=None):
        L = self.S(t1,t0)
        return self.integrate_pressure(setup, pf, t0, t1)/L
    
    def integrate_pressure(self,setup,pf,t0=None,t1=None):
        if t0 is None:
            t0 = self.t0
        if t1 is None:
            t1 = self.t1
        
        def P(T):
            if type(T) is not numpy.ndarray:
               t = numpy.ndarray(T,ndmin=1)
            else:
                t = T
            x, y = self(t)
            u, v, p = compute_arbitrary_flow_mp((x,y),pf,setup)
            return p
        
        def F(T):
            return P(T)*self.I(T)
        
        return scipy.integrate.fixed_quad(F, t0, t1, n=100)[0]
    
    def integrate_normal_flow(self,setup,pf,t0=None,t1=None):
        if t0 is None:
            t0 = self.t0
        if t1 is None:
            t1 = self.t1
            
        def J(T):
            if type(T) is not numpy.ndarray:
                t = numpy.ndarray(T,ndmin=1)
            else:
                t = T
            x, y = self(t)
            u, v, p = compute_arbitrary_flow_mp((x,y),pf,setup)
            Nx, Ny = self.N(t)
            return (u*Nx + v*Ny)*self.I(t)
        
        return scipy.integrate.fixed_quad(J, t0, t1, n=100)[0]
        
    def compute_flux(self,setup,pf,t0=None,t1=None):
        if t0 is None:
            t0 = self.t0
        if t1 is None:
            t1 = self.t1
        Jtot = self.integrate_normal_flow(setup,pf,t0,t1)
        L = self.S(t1,t0)
        return Jtot/L
    
    # Returns a new curve object given by (x1(t), y1(t)),
    # where x1(t) = x0(t) + b*n_x(t) and y1(t) = y0(t) + b*n_y(t),
    # where (x0(t), y0(t)) is the current curve
    def off_curve(self,b):
        t0 = self.t0
        t1 = self.t1
        
        X0 = self.X
        Y0 = self.Y
        
        def X1(t):
            return X0(t) + b*self.N(t)[0]
        
        def Y1(t):
            return Y0(t) + b*self.N(t)[1]
        
        def dNmag(t):
            return -(1.0/(np.sqrt(self.dXdt(t)**2 + self.dYdt(t)**2)**3))*(self.dXdt(t)*self.d2X(t) + self.dYdt(t)*self.d2Y(t))
        
        def dX1(t):
            return self.dXdt(t) - b*(dNmag(t)*self.dYdt(t) + self.I(t)*self.d2Y(t))
        
        def dY1(t):
            return self.dYdt(t) + b*(dNmag(t)*self.dXdt(t) + self.I(t)*self.d2X(t))
        
        C1 = Curve(X1,Y1,t0,t1,dX1,dY1)
        
        return C1
        

class Cell:
    
    a = 1.0
    b = 1.0
    c = 10.0
    
    
    # Curve for front edge of cell
    X_f = lambda t: t
    Y_f = lambda t: 1.0*(np.sqrt(1.0-t+0.00001) + np.sqrt(1.0+t+0.00001) - np.sqrt(2.0))
    dX_f = lambda t: np.ones(np.shape(t))
    dY_f = lambda t: 1.0*(0.5/np.sqrt(1.0+t+0.00001)-0.5/np.sqrt(1.0-t+0.00001))
    d2X_f = lambda t: np.zeros(np.shape(t))
    d2Y_f = lambda t: 1.0*0.25*(-1.0/(np.sqrt(1.0-t+0.00001)**3.0) - 1.0/(np.sqrt(1.0+t+0.00001)**3.0))
    t0_f = -1.0
    t1_f = 1.0
    C_f = Curve(X_f,Y_f,t0_f,t1_f,dX_f,dY_f,d2X_f,d2Y_f)
    L_f = C_f.S(t1_f,t0_f)
    
    # Curve for back edge of cell
    X_b = lambda t: t
    Y_b = lambda t: -1.0*(np.sqrt(1.0-t+0.00001) + np.sqrt(1.0+t+0.00001) - np.sqrt(2.0)) - 0.4*(t**2.0 - 1.0)
    dX_b = lambda t: np.ones(np.shape(t))
    dY_b = lambda t: -1.0*(0.5/np.sqrt(1.0+t+0.00001) - 0.5/np.sqrt(1.0-t+0.00001)) - 0.4*(2.0*t)
    d2X_b = lambda t: np.zeros(np.shape(t))
    d2Y_b = lambda t: -1.0*0.25*(-1.0/(np.sqrt(1.0-t+0.00001)**3.0) - 1.0/(np.sqrt(1.0+t+0.00001)**3.0))
    t0_b = -1.0
    t1_b = 1.0
    C_b = Curve(X_b,Y_b,t0_b,t1_b,dX_b,dY_b,d2X_b,d2Y_b)
    L_b = C_b.S(t1_b,t0_b)
    
    # Curve for right edge of cell
    # UNUSED
    X_r = lambda t: .5*np.ones(np.shape(t))
    Y_r = lambda t: t
    dX_r = lambda t: np.zeros(np.shape(t))
    dY_r = lambda t: np.ones(np.shape(t))
    t0_r = -1.0
    t1_r = 0.0
    C_r = Curve(X_r,Y_r,t0_r,t1_r,dX_r,dY_r)
    L_r = C_r.S(t1_r,t0_r)
    
    # Curve for left edge of cell
    # UNUSED
    X_l = lambda t: -.5*np.ones(np.shape(t))
    Y_l = lambda t: t
    dX_l = lambda t: np.zeros(np.shape(t))
    dY_l = lambda t: np.ones(np.shape(t))
    t0_l = -1.0
    t1_l = 0.0
    C_l = Curve(X_l,Y_l,t0_l,t1_l,dX_l,dY_l)
    L_l = C_l.S(t1_l,t0_l)
    
    # Arclength around whole cell
    L = L_f + L_b

    
    def __init__(self):
        pass
    
    def initialize_points(self,ds):
        (x_f,y_f,t_f) = self.C_f.parameterize(np.round(self.L_f/ds))
        #(x_r,y_r,t_r) = self.C_r.parameterize(np.round(self.L_r/ds))
        (x_b,y_b,t_b) = self.C_b.parameterize(np.round(self.L_b/ds))
        #(x_l,y_l,t_l) = self.C_l.parameterize(np.round(self.L_l/ds))
        
        self.x_f = x_f
        self.y_f = y_f
        self.t_f = t_f
        
        self.x_b = x_b
        self.y_b = y_b
        self.t_b = t_b
        
        #self.x_r = x_r
        #self.y_r = y_r
        #self.t_r = t_r
        
        #self.x_l = x_l
        #self.y_l = y_l
        #self.t_l = t_l

    def initialize_angles(self):
        N_f = self.C_f.N(self.t_f)
        th_f = np.arctan2(-N_f[0],-N_f[1])
        
        N_b = self.C_b.N(self.t_b)
        th_b = np.arctan2(-N_b[0],-N_b[1])
        
        self.N_f = N_f
        self.th_f = th_f
        
        self.N_b = N_b
        self.th_b = th_b

    def initialize_velocities(self):
        cos_f = np.cos(self.th_f)
        cos_b = np.cos(self.th_b)
        
        u_f = -cos_f*self.N_f[0]
        v_f = -cos_f*self.N_f[1]
        
        u_b = -cos_b*self.N_b[0]
        v_b = -cos_b*self.N_b[1]
        
        self.u_f = u_f
        self.v_f = v_f
        
        self.u_b = u_b
        self.v_b = v_b
        
        #self.u_r = 0.0
        #self.v_r = 0.0
        
        #self.u_l = 0.0
        #self.v_l = 0.0
        
    def create_point_force_object(self,setup):
        ds = setup.ds
        self.initialize_points(ds)
        self.initialize_angles()
        self.initialize_velocities()
        
        pf = PointForces()
        pf.add_points(self.x_f,self.y_f,uF=self.u_f,vF=self.v_f)
        pf.add_points(self.x_b,self.y_b,uF=self.u_b,vF=self.v_b)
        #pf.add_points(self.x_r,self.y_r,uF=self.u_r,vF=self.v_r)
        #pf.add_points(self.x_l,self.y_l,uF=self.u_l,vF=self.v_l)
        
        return pf
    
    def get_point_force_object(self):
        pf = PointForces()
        pf.add_points(self.x_f,self.y_f,uF=self.u_f,vF=self.v_f)
        pf.add_points(self.x_b,self.y_b,uF=self.u_b,vF=self.v_b)
        return pf
    
    def get_front_point_force_object(self,setup,N):
        pf = self.get_point_force_object()
        circs = self._generate_front_circles(setup,N)
        pf.add_points(circs[0],circs[1],uF=0.0,vF=0.0)
        pf.xCent = circs[2]
        pf.yCent = circs[3]
        return pf
    
    def get_uniform_circle_point_force_object(self,setup,N):
        pf = self.get_point_force_object()
        circs = self._generate_uniform_circles(setup,N)
        pf.add_points(circs[0],circs[1],uF=0.0,vF=0.0)
        pf.xCent = circs[2]
        pf.yCent = circs[3]
        return pf
    
    # Uses the fact that x(t) = t. Needs to be modified if the
    # parameterization changes.
    def inside(self,x,y):
        return (x >= -1.0)*(x <= 1.0)*(self.C_f.Y(x) >= y)*(self.C_b.Y(x) <= y)
    def outside(self,x,y):
        return 1-self.inside(x,y)
    
    def closest_point(self,x,y):
        x = np.array(x,ndmin=1)
        y = np.array(y,ndmin=1)
        N = len(x)
        
        xx = np.zeros(x.shape)
        yy = np.zeros(x.shape)
        rr = np.zeros(x.shape)
        
        for n in xrange(0,N):
            t_f, r_f = self.C_f.D(x[n], y[n])
            t_b, r_b = self.C_b.D(x[n], y[n])
            
            if r_f <= r_b:
                rr[n] = r_f
                x_f, y_f = self.C_f(t_f)
                xx[n] = x_f
                yy[n] = y_f
            else:
                rr[n] = r_b
                x_b, y_b = self.C_b(t_b)
                xx[n] = x_b
                yy[n] = y_b
        return xx, yy, rr

    def jiggle(self,setup,x,y,tri=None,xx=None,yy=None,rr=None,check=None,edge_gap_factor=20):
        if tri is None:
            tri = delaunay(x,y)
        if xx is None or yy is None or rr is None:
            xx, yy, rr = self.closest_point(x, y)
        if check is None:
            check = self._check_interior_points
        
        a = setup.a
        edge_gap = setup.ds*edge_gap_factor
        
        N = len(x)
        
        circumcenters, edges, tri_points, tri_neighbors = tri
        
        dx = x[edges[:,1]] - x[edges[:,0]]
        dy = y[edges[:,1]] - y[edges[:,0]]
        r  = np.sqrt(dx**2.0 + dy**2.0)
        
        nx = dx/r
        ny = dy/r
        
        F = 1.0/((r/a)**2.0)
        
        fX = np.zeros((N))
        fY = np.zeros((N))
        
        for n in range(0,edges.shape[0]):
            e = edges[n]
            fX[e[0]] = fX[e[0]] - F[n]*nx[n]
            fY[e[0]] = fY[e[0]] - F[n]*ny[n]
            
            fX[e[1]] = fX[e[1]] + F[n]*nx[n]
            fY[e[1]] = fY[e[1]] + F[n]*ny[n]
                
        #for n in range(0,N):
        #    if rr[n] <= edge_gap:
        #        # Compute direction from the boundary to the
        #        # point
        #        dx = x[n] - xx[n]
        #        dy = y[n] - yy[n]
        #        
        #        # Repulsive force on the point near the boundary
        #        fX[n] = fX[n] + np.sign(dx)/((dx/a)**2.0)
        #        fY[n] = fY[n] + np.sign(dy)/((dy/a)**2.0)
                
                
        # We choose a timestep based on how quickly we want
        # overlapping balls to separate. We choose some number
        # k, and then define t so that two points separated
        # by a distance of k*a will move so they are separated
        # by a distance of 2*a in one timestep.
        
        #k = 4.0/3.0
        t = 0.001#(1.0 - 0.5*k)*(k**2)*a
        
        xNew = x + t*fX
        yNew = y + t*fY
        
        I = check(setup, xNew, yNew, edge_gap_factor)
        
        xOut = numpy.where(I, xNew, x)
        yOut = numpy.where(I, yNew, y)
        
        return xOut, yOut
    
    def _random_front_interior_points(self,setup,N,edge_gap_factor=20):
        X, Y = self._random_uniform_points(setup,N)
        X_I, Y_I = self._return_front_points(setup,X,Y)
        if len(X_I) < N:
            X_I_2, Y_I_2 = self._random_front_interior_points(setup,N-len(X_I),edge_gap_factor)
            X_I = numpy.append(X_I,X_I_2)
            Y_I = numpy.append(Y_I,Y_I_2)
        return X_I, Y_I
    
    def _random_uniform_interior_points(self,setup,N,edge_gap_factor=20):
        X, Y = self._random_uniform_points(setup,N)
        X_I, Y_I = self._return_interior_points(setup,X,Y,edge_gap_factor)
        if len(X_I) < N:
            X_I_2, Y_I_2 = self._random_uniform_interior_points(setup,N-len(X_I),edge_gap_factor)
            X_I = numpy.append(X_I,X_I_2)
            Y_I = numpy.append(Y_I,Y_I_2)
        return X_I, Y_I
    
    def _random_uniform_points(self,setup,N):
        x_min = -1.2
        x_max = 1.2
        y_min = -.4
        y_max = 0.6
        
        X = numpy.random.rand((N))*(x_max-x_min) + x_min
        Y = numpy.random.rand((N))*(y_max-y_min) + y_min
        
        return X, Y
    
    def _return_checked_points(self,setup,X,Y,edge_gap_factor=20,closest=None,check=None):
        if check is None:
            check = self._check_interior_points
        
        I = check(setup, X, Y, edge_gap_factor, closest)
        
        X_I = []
        Y_I = []
        for n in xrange(0,len(X)):
            if I[n]:
                X_I.append(X[n])
                Y_I.append(Y[n])
        X_I = numpy.array(X_I)
        Y_I = numpy.array(Y_I)
        return X_I, Y_I
    
    def _return_interior_points(self,setup,X,Y,edge_gap_factor=20,closest=None):
        return self._return_checked_points(setup,X,Y,edge_gap_factor,closest,
                                           check=self._check_interior_points)
    
    def _return_front_points(self,setup,X,Y,edge_gap_factor=20,closest=None):
        return self._return_checked_points(setup,X,Y,edge_gap_factor,closest,
                                           check=self._check_front_points)
        
    def _check_interior_points(self,setup,X,Y,edge_gap_factor=20,closest=None):
        # Prior to doing the super high density, low radius simulations, edge_gap
        # was defined as:
        edge_gap = setup.ds*edge_gap_factor
        
        # Now we define it to scale with radius, and to be roughly consistent with
        # the old definition.  If it is necessary to reproduce
        # the old simulations, it should get switched back to the old definition
        # for consistency.
        # edge_gap = edge_gap_factor*(2*np.pi/25.)*setup.a
        
        
        if closest is None:
            closest = self.closest_point(X,Y)
        xx, yy, rr = closest
        
        I = self.inside(X,Y)*(rr >= edge_gap + setup.a)
        
        return I
    
    def _check_front_points(self,setup,X,Y,edge_gap_factor=20,closest=None):
        max_front_dist = 0.1
        
        # See edge_gap comment in previous function
        edge_gap = setup.ds*edge_gap_factor
        # edge_gap = edge_gap_factor*(2*np.pi/25.)*setup.a
        
        if closest is None:
            closest = self.closest_point(X,Y)
        xx, yy, rr = closest
        
        t_f, r_f = self.C_f.D(X,Y)
        
        y_min = min(self.C_f.Y(-1.0),self.C_f.Y(1.0))
        
        I = self.inside(X,Y)*(rr >= edge_gap + setup.a)*(r_f <= max_front_dist)*(Y >= y_min)
        return I
    
    def _replace_overlapping(self,setup,X,Y,edge_gap_factor=20,tri=None,generate=None):
        if generate is None:
            generate = self._random_uniform_interior_points
        if tri is None:
            tri = delaunay(X,Y)
        circumcenters, edges, tri_points, tri_neighbors = tri
        dx = X[edges[:,1]] - X[edges[:,0]]
        dy = Y[edges[:,1]] - Y[edges[:,0]]
        r = np.sqrt(dx**2.0 + dy**2.0)
        
        overlap_edges = numpy.where(r < 2*setup.a)
        overlaps = numpy.unique(edges[overlap_edges][:,0])
        
        N = max(len(overlaps),1)
        
        xNew, yNew = generate(setup, N, edge_gap_factor)
        
        X[overlaps] = xNew
        Y[overlaps] = yNew
        
        return X, Y
    
    def _check_overlapping(self,setup,X,Y,tri=None):
        if tri is None:
            tri = delaunay(X,Y)
        circumcenters, edges, tri_points, tri_neighbors = tri
        dx = X[edges[:,1]] - X[edges[:,0]]
        dy = Y[edges[:,1]] - Y[edges[:,0]]
        r = np.sqrt(dx**2.0 + dy**2.0)
        
        I = r < 2*setup.a
        return I
    
    def _is_overlapping(self,setup,X,Y,tri=None):
        if tri is None:
            tri = delaunay(X,Y)
        circumcenters, edges, tri_points, tri_neighbors = tri
        dx = X[edges[:,1]] - X[edges[:,0]]
        dy = Y[edges[:,1]] - Y[edges[:,0]]
        r = np.sqrt(dx**2.0 + dy**2.0)
        rmin = min(r)
    
        if rmin >= 2*setup.a:
            return False
        return True
    
    def _generate_circles(self,setup,N,generate):
        X, Y = generate(setup,N)
        circs = generate_circles(setup, 0, 0, withCenters=True, cents=(X,Y))
        return circs
    
    def _generate_front_circles(self,setup,N):
        return self._generate_circles(setup,N,generate=self._generate_front_centers)
    
    def _generate_uniform_circles(self,setup,N):
        return self._generate_circles(setup,N,generate=self._generate_uniform_centers)
    
    def _generate_uniform_centers(self,setup,N):
        return self._generate_centers(setup, N,
                                      generate=self._random_uniform_interior_points,
                                      jiggle_check=self._check_interior_points)
    
    def _generate_front_centers(self,setup,N):
        return self._generate_centers(setup, N,
                                      generate=self._random_front_interior_points,
                                      jiggle_check=self._check_front_points)
    
    def _generate_new_centers(self,setup,N,generate,Xfixed,Yfixed,jiggle_check=None):
        if jiggle_check is None:
            jiggle_check = self._check_interior_points
        max_its = 3000
        
        max_jiggles = 15
        jiggles = 0
        
        X, Y = generate(setup, N)
        Xall = numpy.append(X, Xfixed)
        Yall = numpy.append(Y, Yfixed)
        
        tri = delaunay(Xall,Yall)
        num_overlap = numpy.sum(self._check_overlapping(setup, Xall, Yall, tri))
        
        for i in xrange(0,max_its):
            if jiggles >= max_jiggles:
                Xnew, Ynew = self._replace_overlapping(setup, Xall, Yall, generate=generate)
                Xall[0:len(X)] = Xnew[0:len(X)]
                Yall[0:len(Y)] = Ynew[0:len(Y)]
                replace_points = False
                jiggles = 0
                tri = delaunay(Xall, Yall)
                num_overlap = numpy.sum(self._check_overlapping(setup, Xall, Yall, tri))
            if num_overlap > 0:
                Xnew, Ynew = self.jiggle(setup, Xall, Yall, tri, edge_gap_factor=20, check=jiggle_check)
                Xall[0:len(X)] = Xnew[0:len(X)]
                Yall[0:len(Y)] = Ynew[0:len(Y)]
                num_overlap_old = num_overlap
                tri = delaunay(Xall,Yall)
                num_overlap = numpy.sum(self._check_overlapping(setup, Xall, Yall, tri))
                
                if num_overlap >= num_overlap_old:
                    jiggles += 1
                else:
                    jiggles = 0
            else:
                return Xall[0:len(X)], Yall[0:len(Y)]
        return Xall[0:len(X)], Yall[0:len(Y)] 
                
    
    def _generate_centers(self,setup,N,generate,jiggle_check=None):
        if jiggle_check is None:
            jiggle_check = self._check_interior_points
        max_its = 3000
        
        # Maximum number of jiggles in a row without reducing
        # the number of overlaps
        max_jiggles = 15
        
        # Current number of jiggles at this overlap
        jiggles = 0
        
        # Generate initial set of points in the region interior
        print 'Generating new set of points'
        X, Y = generate(setup, N)
        
        tri = delaunay(X,Y)
        num_overlap = numpy.sum(self._check_overlapping(setup, X, Y, tri))
        
        for i in xrange(0,max_its):
            # If jiggling has stagnated, regenerate overlapping points
            if jiggles >= max_jiggles:
                print 'Replacing overlapping points'
                X, Y = self._replace_overlapping(setup, X, Y, generate=generate)
                replace_points = False
                jiggles = 0
                tri = delaunay(X,Y)
                num_overlap = numpy.sum(self._check_overlapping(setup, X, Y, tri))
                
            
            # Check if all points are spaced far enough apart
            # to be centers of circles each with radius a
            if num_overlap > 0:
                # If not, then jiggle the points
                print str(num_overlap) + ' points overlap: jiggling. Jiggle count is ' + str(jiggles)
                X, Y = self.jiggle(setup, X, Y, tri, edge_gap_factor=20, check=jiggle_check)
                num_overlap_old = num_overlap
                tri = delaunay(X,Y)
                num_overlap = numpy.sum(self._check_overlapping(setup, X, Y, tri))
                
                if num_overlap >= num_overlap_old:
                    jiggles += 1
                else:
                    jiggles = 0
                
            else:
                # If so, then return the points
                print "Points don't overlap. Returning"
                return X, Y
        
            # Check if the jiggle pushed anything outside the cell.
            #I = jiggle_check(setup, X, Y, edge_gap_factor=3)
            #if numpy.mod(i,25) == 0:
            #    print 'Jiggled 25 times. Starting over with new set.'
            #    # If the jiggle pushed things out of the cell, we should
            #    # pick new points
            #    generate_new_points = True
        print 'Reached max iterations. Returning'
        return X, Y

    
# Creates a parameterized curve from two parameterized curves.
# The two curves should both be functions of t defined on the
# intervals (t0_0, t0_1) and (t1_0, t1_1), respectively. The
# resulting curve will be a piecewise function defined on the
# interval (t0_0, t0_1 + t1_1 - t1_0). In words, the function
# still starts at the start of the interval from the (X0, Y0)
# curve, and the interval from the (X1, Y1) curve is shifted
# to connect properly
def link(X0,Y0,t0_0,t0_1,X1,Y1,t1_0,t1_1):
    X1_shift = lambda t: X1(t - t0_1 + t1_0)
    Y1_shift = lambda t: Y1(t - t0_1 + t1_0)
    
    # Define the piecewise function
    X_p = lambda t: np.piecewise(t,[t <= t0_1, t > t0_1],[X0, X1_shift])
    Y_p = lambda t: np.piecewise(t,[t <= t0_1, t > t0_1],[Y0, Y1_shift])
    
    # The above definition requires a numpy array be passed. To make
    # it safe to pass in scalars, we redefine as below
    X = lambda t: X_p(np.array(t,ndmin=1))
    Y = lambda t: Y_p(np.array(t,ndmin=1))
    
    # Limits of the new piecewise curve
    t0 = t0_0
    t1 = t0_1 + t1_1 - t1_0
    
    return (X, Y, t0, t1)

def parameterize(X,Y,dXdt,dYdt,t0,t1,N,centered=True):
    # Integrand for the arclength computation
    I = lambda t: np.sqrt(dXdt(t)**2.0 + dYdt(t)**2.0)
    
    # Arclength S(t) of the parametric curve
    S = lambda t: (scipy.integrate.quad(I,t0,t))[0]
    
    # Inverse T(s) of the arclength
    def T(s):
        # Function whose root we wish to find
        F = lambda t: S(t) - s
        
        t = scipy.optimize.newton(F, s, I, tol=1.0e-14)
        
        return t
    
    # Arclength at endpoints of curve. s0 should be 0
    s0 = S(t0)
    s1 = S(t1)
    
    # Arclength values for the points we will return
    if centered:
        ds = (s1 - s0)/N
        s = np.linspace(s0+0.5*ds,s1-0.5*ds,N,True)
    else:
        s = np.linspace(s0,s1,N)

    t = []
    
    for ss in s:
        tt = T(ss)
        t.append(tt)
        
    t = np.array(t)
    
    return (X(t),Y(t),t)

