from common import *
import scipy.linalg as la
import scipy.sparse as sp
import scipy.sparse.linalg as spla

import scipy.integrate as integrate

from fluidgrid import *

import matplotlib.pyplot as plt

#import pyamg


def vectorize(xComponent,yComponent):
    return np.append(xComponent,yComponent)

def unvectorize(vec):
    size = vec.size
    N = size/2
    xComponent = vec[:N]
    yComponent = vec[N:]
    return xComponent, yComponent

def compute_flow_1D(pf,x,y,setup):
    nF = pf.xF.shape[0]
    nOut = x.shape[0]
    e = setup.e
    mu = setup.mu
    
    P = np.zeros((nOut))
    U = np.zeros((nOut))
    V = np.zeros((nOut))
    
    for k in range(0,nF):
        # x - x_k and y - y_k
        xMxK = x - pf.xF[k]
        yMyK = y - pf.yF[k]
        
        # f dot (x - x_k)
        f_dot_xMxK = pf.fF[k]*xMxK + pf.gF[k]*yMyK
        
        r2 = xMxK**2.0 + yMyK**2.0
        e2 = e**2.0
        r = sqrt(r2)
        root_r2Pe2 = sqrt(r2 + e2)
        
        P = P + (1/(2*pi))*f_dot_xMxK*\
            ((r2 + 2*e2 + e*root_r2Pe2)/((root_r2Pe2 + e)*(root_r2Pe2)**3.0))
        
        velFactor1 = log(root_r2Pe2 + e) - \
            (e*(root_r2Pe2 + 2*e))/((root_r2Pe2 + e)*root_r2Pe2)
        velFactor2 = (root_r2Pe2 + 2*e)/(((root_r2Pe2+e)**2.0)*root_r2Pe2)
        
        U = U + (-pf.fF[k]/(4*pi*mu))*velFactor1 + \
            (1.0/(4*pi*mu))*f_dot_xMxK*xMxK*velFactor2
        
        V = V + (-pf.gF[k]/(4*pi*mu))*velFactor1 + \
            (1.0/(4*pi*mu))*f_dot_xMxK*yMyK*velFactor2
            
    return U, V, P

# Given a PointForces (pf) structure with xF, yF,
# fF, and gF defined, and given a FluidGrid (fg)
# with X and Y defined, computes and stores in
# fg.U, fg.V, and fg.P, the velocity and pressure
# due to those forces
def compute_flow_py(pf,fg,setup):
    nF = pf.xF.shape[0]
    e = setup.e
    mu = setup.mu
    
    for k in range(0,nF):
        # x - x_k and y - y_k
        xMxK = fg.X - pf.xF[k]
        yMyK = fg.Y - pf.yF[k]
        
        # f dot (x - x_k)
        f_dot_xMxK = pf.fF[k]*xMxK + pf.gF[k]*yMyK
        
        r2 = xMxK**2.0 + yMyK**2.0
        e2 = e**2.0
        r = sqrt(r2)
        root_r2Pe2 = sqrt(r2 + e2)
        
        fg.P = fg.P + (1/(2*pi))*f_dot_xMxK*\
            ((r2 + 2*e2 + e*root_r2Pe2)/((root_r2Pe2 + e)*(root_r2Pe2)**3.0))
        
        velFactor1 = log(root_r2Pe2 + e) - \
            (e*(root_r2Pe2 + 2*e))/((root_r2Pe2 + e)*root_r2Pe2)
        velFactor2 = (root_r2Pe2 + 2*e)/(((root_r2Pe2+e)**2.0)*root_r2Pe2)
        
        fg.U = fg.U + (-pf.fF[k]/(4*pi*mu))*velFactor1 + \
            (1.0/(4*pi*mu))*f_dot_xMxK*xMxK*velFactor2
        
        fg.V = fg.V + (-pf.gF[k]/(4*pi*mu))*velFactor1 + \
            (1.0/(4*pi*mu))*f_dot_xMxK*yMyK*velFactor2
    fg.speed = np.sqrt(fg.U**2.0 + fg.V**2.0)

def compute_flow(pf,fg,setup):
    global use_mp
    if use_mp:
        compute_flow_mp_cpp(pf,fg,setup)
    else:
        compute_flow_cpp(pf,fg,setup)
    fg.speed = np.sqrt(fg.U**2.0 + fg.V**2.0)
    
# Compute mobility matrix for a given PointForces
# (pf) class that has the arrays xF and yF defined
def compute_mobility_py(pf,setup):
    Msize = 2*pf.nF
    M = np.zeros((Msize,Msize))
    
    e = setup.e
    mu = setup.mu
    
    for k in range(0,pf.nF):
        x_k = pf.xF[k]
        y_k = pf.yF[k]
        
        fCol = k
        gCol = k + pf.nF
        
        for i in range(0,pf.nF):
            x_i = pf.xF[i]
            y_i = pf.yF[i]
            
            xMxK = x_i - x_k
            yMyK = y_i - y_k
            
            r2 = xMxK**2.0 + yMyK**2.0
            e2 = e**2.0
            r = sqrt(r2)
            root_r2Pe2 = sqrt(r2+e2)
            
            velFactor1 = log(root_r2Pe2 + e) - \
                (e*(root_r2Pe2 + 2*e))/((root_r2Pe2 + e)*root_r2Pe2)
            velFactor2 = (root_r2Pe2 + 2*e)/(((root_r2Pe2+e)**2.0)*root_r2Pe2)
            
            uRow = i
            vRow = i + pf.nF
            
            M[uRow,fCol] = -velFactor1/(4*pi*mu) + (xMxK**2.0)*velFactor2/(4*pi*mu)
            M[uRow,gCol] = xMxK*yMyK*velFactor2/(4*pi*mu)
            M[vRow,fCol] = xMxK*yMyK*velFactor2/(4*pi*mu)
            M[vRow,gCol] = -velFactor1/(4*pi*mu) + (yMyK**2.0)*velFactor2/(4*pi*mu)
    return M

def compute_forces(pf,setup):
    global use_mp
    if use_mp:
        M = compute_mobility_mp_cpp(pf,setup)
        rhs = set_rhs_for_mobility_mp_cpp(pf,setup)
    else:
        M = compute_mobility_cpp(pf,setup)
        rhs = set_rhs_for_mobility_cpp(pf,setup)
    #Fvec, info = spla.gmres(M,rhs)
    #Fvec = la.solve(M,rhs)
    
    #Msp = sp.csr_matrix(M)
    #Fvec, info = spla.cg(Msp,rhs,tol=1e-9)
    
    #print info
    
    #Fvec = pyamg.solve(Msp,rhs,verb=True,tol=1e-8)
    
    Fvec = la.solve(M,rhs)
    
    pf.fF, pf.gF = unvectorize(Fvec)
#    M = compute_mobility(pf,setup)
#    Uvec = vectorize(pf.uF,pf.vF)
#    Fvec = la.solve(M,Uvec)
#    pf.fF, pf.gF = unvectorize(Fvec)

def compute_net_flux(ss,pf,x0,x1,y0,y1,n=100):
    Lx = x1 - x0
    Ly = y1 - y0
    
    fg = FluidGrid(x0-Lx/4.0,x1+Lx/4.0,128,y0-Ly/4.0,y1+Ly/4.0,128)
    
    compute_flow(pf, fg, ss)
    
    F_r = integrate_y_line(x1,y0,y1,pf,ss)
    F_l = integrate_y_line(x0,y0,y1,pf,ss)
    F_t = integrate_x_line(x0,x1,y1,pf,ss)
    F_b = integrate_x_line(x0,x1,y0,pf,ss)
    
    F_net = F_r - F_l + F_t - F_b
    
    plt.contourf(fg.X,fg.Y,fg.speed,500)
    plt.plot((x0,x0),(y0,y1),'r',linewidth=2)
    plt.plot((x1,x1),(y0,y1),'r',linewidth=2)
    plt.plot((x0,x1),(y0,y0),'r',linewidth=2)
    plt.plot((x0,x1),(y1,y1),'r',linewidth=2)
    plt.streamplot(fg.x,fg.y,fg.U,fg.V,density=2,color='white')
    plt.draw()
    
    return F_r, F_l, F_t, F_b, F_net

def integrate_x_line(x0,x1,y0,pf,ss,n=100):
    
    def f(X):
        if type(X) is not np.ndarray:
            x = np.array([X])
        else:
            x = X
        y = y0*np.ones(x.shape)
        u = np.zeros(x.shape)
        v = u.copy()
        p = u.copy()
        compute_arbitrary_flow_mp((x,y),(u,v,p),pf,ss)
        return v
    
    return integrate.fixed_quad(f,x0,x1,args=[],n=n)[0]

def integrate_y_line(x0,y0,y1,pf,ss,n=100):
    
    def f(Y):
        if type(Y) is not np.ndarray:
            y = np.array([Y])
        else:
            y = Y
        x = x0*np.ones(y.shape)
        u = np.zeros(x.shape)
        v = u.copy()
        p = u.copy()
        compute_arbitrary_flow_mp((x,y),(u,v,p),pf,ss)
        return u
    
    return integrate.fixed_quad(f,y0,y1,args=[],n=n)[0]
        
    
    

def gauss_points(a,b,N):
    pass

def gauss(f,a,b,N):
    pass

def compute_exact(fg,setup):
    a = setup.a
    mu = setup.mu
    
    f0 = 8*pi/(1-2*log(a))
    g0 = 0.0
    
    fDotX = f0*fg.X + g0*fg.Y
    
    R2 = fg.X**2.0 + fg.Y**2.0
    R = sqrt(R2)
    
    fg.U = -(f0/(8*pi))*(2*log(R) - (a**2.0)/R2) + (fDotX/(4*pi*R2))*fg.X*(1 - (a**2.0)/R2)
    fg.V = -(g0/(8*pi))*(2*log(R) - (a**2.0)/R2) + (fDotX/(4*pi*R2))*fg.Y*(1 - (a**2.0)/R2)
    
def compute_P_drop(pf,ss,FR_offset=0.01,TB_offset=0.2):
    x_F = 1.0 - FR_offset
    x_R = 0.0 + FR_offset
    y_T = 1.0 - TB_offset
    y_B = 0.0 + TB_offset
    
    y_length = y_T - y_B
    
    def P(x0,Y):
        if type(Y) is not np.ndarray:
            y = np.array([Y])
        else:
            y = Y
        x = x0*np.ones(y.shape)
        u, v, p = compute_arbitrary_flow_mp((x,y),pf,ss)
        return p
    P_F = lambda Y: P(x_F,Y)
    P_R = lambda Y: P(x_R,Y)
    P_F_avg = (integrate.quad(P_F,y_B,y_T,limit=200)[0])/y_length
    P_R_avg = (integrate.quad(P_R,y_B,y_T,limit=200)[0])/y_length
    P_drop = P_F_avg - P_R_avg
    return P_drop
        