from common import *

from pointforces import *
from fluidgrid import *

# Class to handle conversions to and from nondimensional units
class UnitHandler:
    # Initialize length scale (in um), velocity scale (in um/s),
    # and viscosity scale (in pN*s*um^-1).
    def __init__(self, L=10.0, U=0.1, mu=0.01):
        self.L = L
        self.U = U
        self.mu = mu
        
        # Compute the scaling for pressure
        self.P = mu*U/L
        
        # Dictionary of scalings
        S = {}
        
        S['L'] = self.L
        S['X'] = self.L
        S['x'] = self.L
        S['Y'] = self.L
        S['y'] = self.L
        
        S['U'] = self.U
        S['u'] = self.U
        S['V'] = self.U
        S['v'] = self.U
        
        S['P'] = self.P
        S['p'] = self.P
        
        self.S = S
        
    def to_physical(self, A, T):
        return self.S[T]*A
    
    def to_nondimensional(self, A, T):
        return A/self.S[T]
    
    def to_physical_pf(self, pf):
        pf_phys = PointForces()
        pf_phys.xF = self.to_physical(pf.xF, 'x')
        pf_phys.yF = self.to_physical(pf.yF, 'y')
        pf_phys.xCent = self.to_physical(pf.xCent, 'x')
        pf_phys.yCent = self.to_physical(pf.yCent, 'y')
        
        return pf_phys
    
    def to_physical_fg(self, fg):
        fg_phys = FluidGrid(0,0,1,0,0,1)
        fg_phys.P = self.to_physical(fg.P, 'p')
        fg_phys.U = self.to_physical(fg.U, 'u')
        fg_phys.V = self.to_physical(fg.V, 'v')
        fg_phys.X = self.to_physical(fg.X, 'x')
        fg_phys.Y = self.to_physical(fg.Y, 'y')
        fg_phys.x = self.to_physical(fg.x, 'x')
        fg_phys.y = self.to_physical(fg.y, 'y')
        fg_phys.speed = self.to_physical(fg.speed, 'u')
        fg_phys.nX = fg.nX
        fg_phys.nY = fg.nY
        fg_phys.xMin = fg_phys.x.min()
        fg_phys.xMax = fg_phys.x.max()
        fg_phys.yMin = fg_phys.y.min()
        fg_phys.yMax = fg_phys.y.max()
        
        return fg_phys
    