
/*-------------------
Elastic Network By Using Delaunay Triangulation
Units: micron, sec, pN
-------------------*/

#define OPENGL 1	// with graphics or not 
#define SAVPIC 0	// save frames if OPENGL=1

#define EXTNSN 0	// with finger extension or not
#define ECMMLT 1	// with ECM-melting
#define CTXADH 1	// with CTX-adhesion

#define MLTECM 1	// multiple ECM layers or not
#define MEMBND 1	// with membrane bending or not
#define SLFAVD 1	// MEM is self-avoiding or not

#define A_CNST 1	// area is conserved or not (set to 0 for finger)
#define MEMCTX 1	// include mem-node vs. ctx-edge interation (better looking)
#define EMBED_ 1	// embed cell 0=none, 1=clear nucleus, 2=clear cell

#define REPEAT 1	// do repeating test if OPENGL=0
#define NOISE_ 0	// network has noise or not

#define SEARCH 1	// do parameter search if OPENGL=0: 0=no, 1=yes
#define SRCHNO 1	// number of searching parameters

#define STRESS 0	// show stress color or not
#define STRAIN 0	// show strain field or not
#define SHWMLT 1	// show melting region or not
#define PAUSE_ 0	// pause on GL

#define SMDOTS 0	// display smaller dots for membrane nodes
#define AALIAS 1	// anti-alias
#define VCDBUG 0	// debugging mode under VC6 (press F5 to run)

#include "include.h"


void getmseed(void)
{
#if OPENGL && SAVPIC
	mseed=-1;
#else
	struct tm *newtime;
	time_t long_time;

	time(&long_time);	// RNG seed
	newtime=localtime(&long_time);
	mseed=-(int) fabs((newtime->tm_hour+1)*(2*newtime->tm_min+3)*(111*newtime->tm_sec+5)+
		191*(newtime->tm_min)+901*(newtime->tm_sec)-1);
	//mseed=-1;
#endif
}


void init(void)
{
#if REPEAT
	FILE *fp;
#endif

	readpara();
	showpara();
	getconst();

	frepeatsum=frepeatave=0.0;
	vrepeatsum=vrepeatave=0.0;
	repeatcnt=0;
	TIP_ID=0;
	NrepeatValid=0;

	connectednodes=malloc(N_ECM*sizeof(int));	// for solving relaxation
	noise=NOISE_;

	frepeatind=malloc(Nrepeat*sizeof(double));
	vrepeatind=malloc(Nrepeat*sizeof(double));

#if REPEAT
	printf("Run\t<V>\n");
	printf("----------------\n");
	fp=fopen("multi.dat", "w");
	fprintf(fp, "#N\tV(um/hr)\n");
	fclose(fp);
#endif

	// GL
#if OPENGL && SAVPIC
	showskip=5;
	img_cnt=0;
	img_cntm=10;	// skip time = dt*showskip*img_cntm
	nframe=0;
#else
	showskip=10;	// show frame every this iterations
#endif
	pause=PAUSE_;
	showforce=0;
	shownormal=0;
	showstrain=STRAIN;
	showmelt=SHWMLT;
	arrowheadlen=(float)(0.3*ksiECM);
	linkcolor=STRESS;

}


void initview(void)
{
	PanX=PanY=0.0f;
	zoom=(float)(ZOOM*(min2(1.0/(XLEN+R_MEMdb), HWratio/(YLEN+R_MEMdb))));
	zoominv=1.0f/zoom;
}


void CleanUp(void)
{
	CleanECM();
	CleanMEM();
	CleanCTX();
#if VCDBUG
	_CrtDumpMemoryLeaks();
#endif
}


void BuildNetwork(void)
{
	BuildMEM();
	BuildCTX();
	BuildECM();
}



void reset(void)
{
	int i;
	FILE *fp;

	t=-0.02*Tmax;	// to skip the initial adjustment stage
	staton=0;
	N_ADH=0;
	ADH_OK=1;
	smpcnt=smptot=vsmpcnt=0;
	fnucave=vnucave=vcellave=0.0;
	probnuc=probnuctip=probdis=0.0;
	n_track=0;
	Vcell_ave=0;
	ValidRun=1;

	veczero(Fnuc);
	veczero(Vnuc);
	veczero(Fnucave);
	veczero(Vnucave);
	veczero(Vcellave);

	Rnuc[0]=-XLENhalf+2.0*max2(R_MEM, ksiECM);	//XLENhalf-ksiECM-2*R_NUC;	// self-contraction
	Rnuc[1]=0.0;	//ksiECM*0.2*(1-2*ran2(&mseed));	//0.0;
	veccopy(Rnuc, Rnucprev);
	veccopy(Rnuc, Rnucini);

	// path sampling
	smp_cnt=-1;	//smp_skip;	// to avoid recording the initial jump near t=0
	smp_num=smp_ptr=0;
	for(i=0; i<smp_buffer; i++) {
		smp_t_buffer[i]=smp_x_buffer[i]=smp_y_buffer[i]=0;
	}
	fp=fopen("track.dat","w");
	fprintf(fp, "#t\tx\ty\n");
	fclose(fp);

	CleanUp();
	BuildNetwork();
}


void Motion(void)
{
	int i;
	
	for(i=0; i<N_Relax; i++) {
		GetAllForces();
		GetAllMotions();
	}
	if(noise) GetAllFluctuations();
}


void DoSimu(void)
{
	CTXDynamics();
	ECMDynamics();

	Motion();
	if(t>=0) SaveTracking();	// in motion.c

	if(++vsmpcnt >= smpskip) {
		getVnuc();
		vsmpcnt=0;
	}

	if(Xmax<RightBnd && Xmin>LeftBnd) {
		staton=1;
		if(++smpcnt >= smpskip) {
			smpcnt=0;
			smptot++;
			DoSampling();	// make sure smptot++ is in front
		}
	}
	else if(Xmax>RightBnd) staton=0;
	t+=dt;

	if(t<0) {
		if(Rnuc[0]>Rnucini[0]+R_NUC) t=0;	// skip the initial adj. if cell moves
	}
}



void MultipleRuns(void)
{
	FILE *fp;
	
	NrepeatValid=0;
	for(repeatcnt=0; repeatcnt<Nrepeat; repeatcnt++) {
		while(Xmax<RightBnd && t<Tmax && ValidRun) DoSimu();
		if(ValidRun) {	// no membrane crossings
			ProcessTrack();	// get <Vcell>
			vrepeatind[NrepeatValid]=Vcell_ave;
			vrepeatsum+=Vcell_ave;
			printf("%d\t%.3g\n", NrepeatValid+1, Vcell_ave*3600);

			fp=fopen("multi.dat", "a");
			fprintf(fp, "%d\t%.4g\n", NrepeatValid+1, Vcell_ave*3600);
			fclose(fp);
			
			NrepeatValid++;
		}
		if(repeatcnt<Nrepeat-1) reset();	// make sure smptot!=0
	}
}


void terminate(void)
{
	DoStat();
	if(connectednodes) free(connectednodes);
	free(frepeatind);
	free(vrepeatind);
	exit(0);
}


int main(int argc, char **argv)
{
#if OPENGL
	getmseed();
#else
	if(argc==1) {	// no input
		getmseed();
	#if SEARCH
		printf("\nIncorrect searching parameters:\n");
		printf("SRCHNO=%d,\targc=%d\n", SRCHNO, argc);
		//printf("Press Enter to exit...");
		//(void) getchar();
		exit(0);
	#endif
	}
	else {
		mseed=-abs(atoi(argv[1]));	// 1st input: random seed
		if(argc>=2) {	// has more than 1 input
			if(SRCHNO==1 && argc==4) {	// search 1 parameter
				Nsrch1=atoi(argv[2]);	// 2nd input: total # of grids for para1
				ThisSrch=atoi(argv[3]);	// 3rd input: id for this search
			}
			else if(SRCHNO==2 && argc==5) {	// search 2 parameters
				Nsrch1=atoi(argv[2]);	// 2nd input: total # of grids for para1
				Nsrch2=atoi(argv[3]);	// 3nd input: total # of grids for para2
				ThisSrch=atoi(argv[4]);	// 4th input: id for this search
			}
			else {
				printf("\nIncorrect searching parameters:\n");
				printf("SRCHNO=%d,\targc=%d\n", SRCHNO, argc);
				//printf("Press Enter to exit...");
				//(void) getchar();
				exit(0);
			}
		}
	}
#endif

	init();
	initview();
	reset();

#if OPENGL
	RunGL(&argc,argv);
#else
	#if REPEAT
		MultipleRuns();
	#else
		while(t<=Tmax) DoSimu();
		ProcessTrack();
	#endif
	terminate();
#endif


#if VCDBUG
	_CrtDumpMemoryLeaks();
#endif

	return 0;
}

